﻿namespace Hims.Api.Controllers
{
    using Hims.Api.Models;
    using Hims.Domain.Entities;
    using Hims.Domain.Services;
    using Hims.Shared.DataFilters;
    using Hims.Shared.EntityModels;
    using Hims.Shared.Library.Enums;
    using Hims.Shared.UserModels;
    using Hims.Shared.UserModels.Filters;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/callhistorydata")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class CallHistoryController : BaseController
    {
        /// <summary>
        /// The coupon services.
        /// </summary>
        private readonly ICallHistoryDataService callOutDataService;

        /// <inheritdoc />
        public CallHistoryController(ICallHistoryDataService callOutDataService, IAuditLogService auditLogServices)
        {
            this.callOutDataService = callOutDataService;
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<Call_HistoryModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] Call_HistoryModel model, [FromHeader] LocationHeader header)
        {
            model = (Call_HistoryModel)EmptyFilter.Handler(model);
            var beds = await this.callOutDataService.FetchAsync(model);
            return beds == null ? this.ServerError() : this.Success(beds);
        }

        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon added successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] CallHistory model, [FromHeader] LocationHeader header)
        {
            model = (CallHistory)EmptyFilter.Handler(model);
            var response = await this.callOutDataService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given CallHistory has already been exists with us.");
                case 0:
                    return this.ServerError();
            }
            return this.Success("CallHistory has been added successfully.");
        }

        /// <summary>
        /// The delete specialization.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Specialization deleted successfully.
        /// - 409 - Specialization can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] CallHistory model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (CallHistory)EmptyFilter.Handler(model);
                var response = await this.callOutDataService.DeleteAsync(model.Id);
                if (response == 0)
                {
                    return this.ServerError();
                }


                return this.Success("CallHistory has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
                if (exception.Message.Contains("violates foreign key constraint"))
                {
                    return this.Conflict("CallHistory can't be deleted. Please contact Administrator.");
                }

                return this.ServerError();
            }
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetchcallhistoryId")]
        [ProducesResponseType(typeof(List<Call_HistoryModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindCallHistoryById([FromBody] Call_HistoryModel model)
        {
            model = (Call_HistoryModel)EmptyFilter.Handler(model);
            var wards = await this.callOutDataService.FindByMobileNum(model);
            return wards == null ? this.ServerError() : this.Success(wards);
        }
        /// <summary>
        /// The update coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon updated successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] CallHistory model, [FromHeader] LocationHeader header)
        {
            model = (CallHistory)EmptyFilter.Handler(model);
            var response = await this.callOutDataService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("CallHistory has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            return this.Success("CallHistory has been updated successfully.");
        }
    }
}
